using System;
using System.Xml;
using System.Collections;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// Base abstract class for RPC parameters used in RPC request. 
	/// </summary>
	public abstract class RpcParameter
	{
		private readonly ParameterPosition _position;
		private readonly string _typeName;

		/// <summary>
		/// RPC parameter value. Derived members should provide casting 
		/// and XML serialization/deserialization to their specific types.
		/// </summary>
		private object _baseValue;

		// Constants used in XML message serialization/deserialization.
		private const string XMLCONSTS_PARAMETER_NODE_NAME = "Param";
		private const string XMLCONSTS_POSITION_ATTRIBUTE_NAME = "position";
		private const string XMLCONSTS_TYPE_PARAM_ATTRIBUTE_NAME = "type";

		/// <summary>
		/// The only constructor available for derived classes. 
		/// </summary>
		/// <param name="paramTypeName">RPC parameter RPC type name.</param>
		/// <param name="paramPosition">RPC parameter position in remote procedure's signature.</param>
		/// <param name="paramValue">RPC parameter value.</param>
		protected RpcParameter( string paramTypeName, ParameterPosition paramPosition,  object paramValue )
		{
			if( paramTypeName == null )
				throw( new ArgumentNullException( "paramTypeName" ) );

			if( paramPosition == null )
				throw( new ArgumentNullException( "paramPosition" ) );

			_typeName = paramTypeName;			
			_position = paramPosition;
			_baseValue = paramValue;
		}

		/// <summary>
		/// Deserialization constructor. Creates instance of class from XML, 
		/// previously created by deserialization of object of the same class. 
		/// </summary>
		/// <param name="paramTypeName">RPC parameter RPC type name.</param>
		/// <param name="parameterNode">XML node to parse and get parameter information from.</param>
		protected RpcParameter( string paramTypeName, XmlNode parameterNode )
		{
			if( parameterNode == null )
				throw( new ArgumentNullException( "parameterNode" ) );

			XmlElement _parameterElement = XmlParseGetParameterElement( parameterNode );
		
			XmlUtility.ParseCheckRequiredAttributeValue( _parameterElement, XMLCONSTS_TYPE_PARAM_ATTRIBUTE_NAME, paramTypeName );
			
			_typeName = paramTypeName;
			_position = ParameterPosition.Parse( XmlUtility.ParseGetRequiredAttributeValue( _parameterElement, XMLCONSTS_POSITION_ATTRIBUTE_NAME ) );
			_baseValue = ParseParameterValue( _parameterElement );
		}		

		/// <summary>
		/// XML serialization method writing out XML representation of RPC parameter.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		public void WriteParameterToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_PARAMETER_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_TYPE_PARAM_ATTRIBUTE_NAME, _typeName );
			writer.WriteAttributeString( XMLCONSTS_POSITION_ATTRIBUTE_NAME, _position.ToString() );

			if( _baseValue == null )
				WriteParameterNullValue( writer );
			else
				WriteParameterValue( writer );
			
			writer.WriteEndElement();
		}

		/// <summary>
		/// XML serialization method writing out parameter value. 
		/// Derived classes must override this method to be XML serializable.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected abstract void WriteParameterValue( XmlWriter writer );

		/// <summary>
		/// XML deserialization method reading parameter value.
		/// Child classes must override this method to parse parameter value.
		/// </summary>
		/// <param name="parameterElement">
		///		Parameter element. Parameter value resides in a child node of this element.
		///	</param>
		protected abstract object ParseParameterValue( XmlElement parameterElement );

		/// <summary>
		/// This method will be called during XML serialization 
		/// instead of WriteParameterValue if RPC parameter value is null. 
		/// Basically, it doesn't write anything, leaving value in XML blank. 
		/// Derived classes may override this method to introduce different behavior.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected virtual void WriteParameterNullValue( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteString( GlobalConfig.RpcXmlNullValue	);
		}		

		/// <summary>
		/// XML parsing utility method used to convert parameter node into XmlElement.
		/// </summary>
		/// <param name="testNode">Parameter node to convert.</param>
		/// <returns>Reference to a given node cast to XmlElement.</returns>
		protected static XmlElement XmlParseGetParameterElement( XmlNode testNode )
		{
			if( testNode == null )
				throw( new ArgumentNullException( "testNode" ) );

			return XmlUtility.ParseValidateConvertNodeToElement( testNode, XMLCONSTS_PARAMETER_NODE_NAME );
		}

		/// <summary>
		/// This method is used during XML deserialization to recognize parameter XML.
		/// It extracts and returns parameter type string name from supplied RPC parameter node.
		/// </summary>
		/// <param name="parameterNode">Parameter node to extract type name from.</param>
		/// <returns>String containing parameter type name.</returns>
		public static string XmlParseGetParameterTypeName( XmlNode parameterNode )
		{
			return XmlUtility.ParseGetRequiredAttributeValue( XmlParseGetParameterElement( parameterNode ), XMLCONSTS_TYPE_PARAM_ATTRIBUTE_NAME );
		}

		/// <summary>
		/// RPC parameter position in remote procedure signature. 
		/// </summary>
		public ParameterPosition Position
		{
			get
			{
				return _position;
			}
		}

		/// <summary>
		/// This property provides unified read-only access to RPC parameter's base .NET value.
		/// </summary>
		public object BaseValue 
		{
			get
			{
				return _baseValue;
			}
		}

		/// <summary>
		/// Enables derived classes to set parameter value.
		/// </summary>
		/// <param name="valueToSet">Value to assign to RPC parameter.</param>
		protected void SetBaseValue( object valueToSet )
		{
			_baseValue = valueToSet;
		}

		/// <summary>
		/// RPC parameter's underlying .NET type. 
		/// </summary>
		public abstract Type BaseType
		{
			get;
		}
	}
}
